/**
 * Background service worker for Sortula extension
 * Handles context menus, offline queue sync, and notifications
 */
import { api } from './shared/api.js';
import { storage } from './shared/storage.js';
import { config } from './shared/config.js';
import { getMessage } from './shared/i18n.js';

// Context menu IDs
const MENU_QUICK_SAVE = 'sortula-quick-save';
const MENU_CONFIGURE_SAVE = 'sortula-configure-save';
const MENU_SAVE_SELECTION = 'sortula-save-selection';

/**
 * Initialize extension on install/update
 */
chrome.runtime.onInstalled.addListener(async () => {
  console.log('Sortula extension installed/updated');
  await setupContextMenus();
});

/**
 * Setup context menus
 */
async function setupContextMenus() {
  // Remove existing menus
  await chrome.contextMenus.removeAll();

  const isAuthenticated = await storage.isAuthenticated();
  if (!isAuthenticated) {
    return;
  }

  // Quick save current page
  chrome.contextMenus.create({
    id: MENU_QUICK_SAVE,
    title: getMessage('contextMenuQuickSave'),
    contexts: ['page', 'link'],
  });

  // Configure and save
  chrome.contextMenus.create({
    id: MENU_CONFIGURE_SAVE,
    title: getMessage('contextMenuConfigureSave'),
    contexts: ['page', 'link'],
  });

  // Save selected text as quote
  chrome.contextMenus.create({
    id: MENU_SAVE_SELECTION,
    title: getMessage('contextMenuSaveSelection'),
    contexts: ['selection'],
  });
}

/**
 * Handle context menu clicks
 */
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  const url = info.linkUrl || info.pageUrl || tab.url;
  const title = tab.title || url;

  switch (info.menuItemId) {
    case MENU_QUICK_SAVE:
      await quickSave(url, title, tab);
      break;

    case MENU_CONFIGURE_SAVE:
      // Open popup with pre-filled data for configuration
      await chrome.storage.local.set({
        pendingBookmark: { url, title, mode: 'configure' },
      });
      chrome.action.openPopup();
      break;

    case MENU_SAVE_SELECTION:
      await saveSelection(url, title, info.selectionText, tab);
      break;
  }
});

/**
 * Quick save bookmark without confirmation
 */
async function quickSave(url, title, tab) {
  try {
    // Check if online
    if (!navigator.onLine) {
      await addToOfflineQueue({ url, title });
      showNotification(
        getMessage('notificationOfflineTitle'),
        getMessage('notificationOfflineMessage')
      );
      return;
    }

    await api.createBookmark({ url, title });

    showNotification(
      getMessage('notificationSavedTitle'),
      getMessage('notificationSavedMessage', [title])
    );
  } catch (err) {
    console.error('Quick save failed:', err);

    // If network error, add to offline queue
    if (err.message.includes('Failed to fetch') || err.message.includes('NetworkError')) {
      await addToOfflineQueue({ url, title });
      showNotification(
        getMessage('notificationOfflineTitle'),
        getMessage('notificationOfflineMessage')
      );
    } else {
      showNotification(
        getMessage('notificationErrorTitle'),
        err.message
      );
    }
  }
}

/**
 * Save selected text as a quote
 */
async function saveSelection(url, title, selectedText, tab) {
  try {
    const bookmarkData = {
      url,
      title,
      note: selectedText,
    };

    if (!navigator.onLine) {
      await addToOfflineQueue(bookmarkData);
      showNotification(
        getMessage('notificationOfflineTitle'),
        getMessage('notificationOfflineMessage')
      );
      return;
    }

    await api.createBookmark(bookmarkData);

    showNotification(
      getMessage('notificationQuoteSavedTitle'),
      getMessage('notificationQuoteSavedMessage')
    );
  } catch (err) {
    console.error('Save selection failed:', err);
    showNotification(
      getMessage('notificationErrorTitle'),
      err.message
    );
  }
}

/**
 * Add bookmark to offline queue
 */
async function addToOfflineQueue(bookmarkData) {
  const queue = await storage.addToOfflineQueue(bookmarkData);
  updateBadge(queue.length);
}

/**
 * Sync offline queue when back online
 */
async function syncOfflineQueue() {
  const queue = await storage.getOfflineQueue();
  if (queue.length === 0) return;

  console.log(`Syncing ${queue.length} offline bookmarks...`);

  for (const item of queue) {
    try {
      await api.createBookmark({
        url: item.url,
        title: item.title,
        note: item.note,
      });
      await storage.removeFromOfflineQueue(item.id);
    } catch (err) {
      console.error(`Failed to sync bookmark ${item.id}:`, err);
      // Stop syncing if we hit an error (might be offline again)
      break;
    }
  }

  const remaining = await storage.getOfflineQueue();
  updateBadge(remaining.length);

  if (remaining.length === 0) {
    showNotification(
      getMessage('notificationSyncCompleteTitle'),
      getMessage('notificationSyncCompleteMessage')
    );
  }
}

/**
 * Update extension badge with pending count
 */
function updateBadge(count) {
  if (count > 0) {
    chrome.action.setBadgeText({ text: count.toString() });
    chrome.action.setBadgeBackgroundColor({ color: '#FF5C00' });
  } else {
    chrome.action.setBadgeText({ text: '' });
  }
}

/**
 * Show browser notification
 */
function showNotification(title, message) {
  chrome.notifications.create({
    type: 'basic',
    iconUrl: 'icons/icon128.png',
    title,
    message,
  });
}

// Listen for online/offline events
self.addEventListener('online', () => {
  console.log('Back online, syncing...');
  syncOfflineQueue();
});

// Listen for messages from popup/content scripts
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === 'QUICK_SAVE') {
    quickSave(message.url, message.title, sender.tab).then(() => {
      sendResponse({ success: true });
    }).catch((err) => {
      sendResponse({ success: false, error: err.message });
    });
    return true; // Keep channel open for async response
  }

  if (message.type === 'AUTH_CHANGED') {
    setupContextMenus();
    sendResponse({ success: true });
    return false;
  }

  if (message.type === 'GET_CURRENT_TAB') {
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      sendResponse({ tab: tabs[0] });
    });
    return true;
  }
});

// Periodic sync check
setInterval(async () => {
  if (navigator.onLine) {
    const queue = await storage.getOfflineQueue();
    if (queue.length > 0) {
      syncOfflineQueue();
    }
  }
}, config.SYNC_INTERVAL);

// Update badge on startup
storage.getOfflineQueue().then((queue) => {
  updateBadge(queue.length);
});
