/**
 * Sortula Extension Popup
 */
import { api } from '../shared/api.js';
import { storage } from '../shared/storage.js';
import { getMessage } from '../shared/i18n.js';

// DOM Elements
const screens = {
  loading: document.getElementById('loading'),
  activation: document.getElementById('activation'),
  main: document.getElementById('main'),
};

const elements = {
  activationCode: document.getElementById('activationCode'),
  activateBtn: document.getElementById('activateBtn'),
  activationError: document.getElementById('activationError'),
  userEmail: document.getElementById('userEmail'),
  logoutBtn: document.getElementById('logoutBtn'),
  bookmarkTitle: document.getElementById('bookmarkTitle'),
  bookmarkUrl: document.getElementById('bookmarkUrl'),
  bookmarkNote: document.getElementById('bookmarkNote'),
  saveBtn: document.getElementById('saveBtn'),
  mainError: document.getElementById('mainError'),
  mainSuccess: document.getElementById('mainSuccess'),
  offlineQueue: document.getElementById('offlineQueue'),
  offlineCount: document.getElementById('offlineCount'),
};

/**
 * Initialize popup
 */
async function init() {
  // Apply translations
  applyTranslations();

  // Check authentication
  const isAuthenticated = await storage.isAuthenticated();

  if (isAuthenticated) {
    await showMainScreen();
  } else {
    showActivationScreen();
  }

  // Setup event listeners
  setupEventListeners();
}

/**
 * Apply i18n translations to elements
 */
function applyTranslations() {
  document.querySelectorAll('[data-i18n]').forEach(el => {
    const key = el.getAttribute('data-i18n');
    el.textContent = getMessage(key);
  });

  document.querySelectorAll('[data-i18n-placeholder]').forEach(el => {
    const key = el.getAttribute('data-i18n-placeholder');
    el.placeholder = getMessage(key);
  });
}

/**
 * Show screen by ID, hide others
 */
function showScreen(screenId) {
  Object.entries(screens).forEach(([id, el]) => {
    el.classList.toggle('hidden', id !== screenId);
  });
}

/**
 * Show activation screen
 */
function showActivationScreen() {
  showScreen('activation');
  elements.activationCode.focus();
}

/**
 * Show main screen with current tab info
 */
async function showMainScreen() {
  showScreen('main');

  // Load user info
  const user = await storage.getUser();
  if (user) {
    elements.userEmail.textContent = user.email;
  }

  // Get current tab
  const tab = await getCurrentTab();
  if (tab) {
    elements.bookmarkTitle.value = tab.title || '';
    elements.bookmarkUrl.value = tab.url || '';
  }

  // Check for pending bookmark from context menu
  const { pendingBookmark } = await chrome.storage.local.get('pendingBookmark');
  if (pendingBookmark) {
    elements.bookmarkTitle.value = pendingBookmark.title || elements.bookmarkTitle.value;
    elements.bookmarkUrl.value = pendingBookmark.url || elements.bookmarkUrl.value;
    await chrome.storage.local.remove('pendingBookmark');
  }

  // Update offline queue indicator
  await updateOfflineQueueIndicator();
}

/**
 * Get current active tab
 */
async function getCurrentTab() {
  return new Promise((resolve) => {
    chrome.runtime.sendMessage({ type: 'GET_CURRENT_TAB' }, (response) => {
      resolve(response?.tab || null);
    });
  });
}

/**
 * Update offline queue indicator
 */
async function updateOfflineQueueIndicator() {
  const queue = await storage.getOfflineQueue();
  if (queue.length > 0) {
    elements.offlineQueue.classList.remove('hidden');
    elements.offlineCount.textContent = getMessage('offlineQueueCount', [queue.length.toString()]);
  } else {
    elements.offlineQueue.classList.add('hidden');
  }
}

/**
 * Setup event listeners
 */
function setupEventListeners() {
  // Activation code input - auto uppercase and focus next
  elements.activationCode.addEventListener('input', (e) => {
    e.target.value = e.target.value.toUpperCase().replace(/[^A-Z0-9]/g, '');
  });

  // Activate button
  elements.activateBtn.addEventListener('click', handleActivate);

  // Enter key on activation code
  elements.activationCode.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') {
      handleActivate();
    }
  });

  // Logout button
  elements.logoutBtn.addEventListener('click', handleLogout);

  // Save button
  elements.saveBtn.addEventListener('click', handleSave);
}

/**
 * Handle activation
 */
async function handleActivate() {
  const code = elements.activationCode.value.trim();

  if (code.length !== 6) {
    showError(elements.activationError, getMessage('activationError'));
    return;
  }

  elements.activateBtn.disabled = true;
  elements.activateBtn.querySelector('span').textContent = getMessage('activationActivating');
  hideError(elements.activationError);

  try {
    await api.activate(code);

    // Notify background script
    chrome.runtime.sendMessage({ type: 'AUTH_CHANGED' });

    // Show main screen
    await showMainScreen();
  } catch (err) {
    showError(elements.activationError, getMessage('activationError'));
    elements.activateBtn.disabled = false;
    elements.activateBtn.querySelector('span').textContent = getMessage('activationButton');
  }
}

/**
 * Handle logout
 */
async function handleLogout() {
  await api.logout();

  // Notify background script
  chrome.runtime.sendMessage({ type: 'AUTH_CHANGED' });

  // Show activation screen
  showActivationScreen();
  elements.activationCode.value = '';
}

/**
 * Handle save bookmark
 */
async function handleSave() {
  const url = elements.bookmarkUrl.value.trim();
  const title = elements.bookmarkTitle.value.trim();
  const note = elements.bookmarkNote.value.trim();

  if (!url) {
    showError(elements.mainError, 'URL is required');
    return;
  }

  elements.saveBtn.disabled = true;
  elements.saveBtn.classList.add('saving');
  elements.saveBtn.querySelector('span').textContent = getMessage('popupSaving');
  hideError(elements.mainError);
  hideSuccess(elements.mainSuccess);

  try {
    await api.createBookmark({ url, title, note: note || undefined });

    elements.saveBtn.classList.remove('saving');
    elements.saveBtn.classList.add('saved');
    elements.saveBtn.querySelector('span').textContent = getMessage('popupSaved');

    showSuccess(elements.mainSuccess, getMessage('notificationSavedMessage', [title]));

    // Close popup after short delay
    setTimeout(() => {
      window.close();
    }, 1500);
  } catch (err) {
    // Check if offline
    if (!navigator.onLine || err.message.includes('Failed to fetch')) {
      // Add to offline queue
      await storage.addToOfflineQueue({ url, title, note });
      await updateOfflineQueueIndicator();

      elements.saveBtn.classList.remove('saving');
      elements.saveBtn.classList.add('saved');
      elements.saveBtn.querySelector('span').textContent = getMessage('popupSaved');

      showSuccess(elements.mainSuccess, getMessage('notificationOfflineMessage'));

      setTimeout(() => {
        window.close();
      }, 1500);
    } else {
      showError(elements.mainError, err.message);
      elements.saveBtn.disabled = false;
      elements.saveBtn.classList.remove('saving');
      elements.saveBtn.querySelector('span').textContent = getMessage('popupSaveButton');
    }
  }
}

/**
 * Show error message
 */
function showError(element, message) {
  element.textContent = message;
  element.classList.remove('hidden');
}

/**
 * Hide error message
 */
function hideError(element) {
  element.classList.add('hidden');
}

/**
 * Show success message
 */
function showSuccess(element, message) {
  element.textContent = message;
  element.classList.remove('hidden');
}

/**
 * Hide success message
 */
function hideSuccess(element) {
  element.classList.add('hidden');
}

// Initialize on load
document.addEventListener('DOMContentLoaded', init);
