/**
 * Sortula API client for browser extension
 */
import { config } from './config.js';
import { storage } from './storage.js';

class ApiClient {
  constructor() {
    this.baseUrl = config.API_URL;
  }

  /**
   * Make authenticated API request
   */
  async request(endpoint, options = {}) {
    const tokens = await storage.getTokens();

    // Check if token needs refresh
    if (tokens?.accessToken) {
      const tokenData = this.parseJwt(tokens.accessToken);
      const expiresIn = (tokenData.exp * 1000) - Date.now();

      if (expiresIn < config.TOKEN_REFRESH_THRESHOLD && tokens.refreshToken) {
        try {
          await this.refreshTokens();
        } catch (err) {
          console.error('Token refresh failed:', err);
          // Continue with existing token, it might still work
        }
      }
    }

    const headers = {
      'Content-Type': 'application/json',
      ...options.headers,
    };

    // Get fresh tokens after potential refresh
    const freshTokens = await storage.getTokens();
    if (freshTokens?.accessToken) {
      headers['Authorization'] = `Bearer ${freshTokens.accessToken}`;
    }

    const response = await fetch(`${this.baseUrl}${endpoint}`, {
      ...options,
      headers,
    });

    if (!response.ok) {
      const error = await response.json().catch(() => ({}));
      throw new Error(error.detail || `HTTP ${response.status}`);
    }

    return response.json();
  }

  /**
   * Activate extension with 6-character code
   */
  async activate(code) {
    const response = await fetch(`${this.baseUrl}/auth/extension/activate`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ code: code.toUpperCase() }),
    });

    if (!response.ok) {
      const error = await response.json().catch(() => ({}));
      throw new Error(error.detail || 'Activation failed');
    }

    const data = await response.json();

    // Store tokens
    await storage.setTokens({
      accessToken: data.access_token,
      refreshToken: data.refresh_token,
    });

    // Store user info
    await storage.setUser(data.user);

    return data;
  }

  /**
   * Refresh access token
   */
  async refreshTokens() {
    const tokens = await storage.getTokens();
    if (!tokens?.refreshToken) {
      throw new Error('No refresh token');
    }

    const response = await fetch(`${this.baseUrl}/auth/refresh`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ refresh_token: tokens.refreshToken }),
    });

    if (!response.ok) {
      // Refresh failed, clear tokens
      await storage.clearAuth();
      throw new Error('Token refresh failed');
    }

    const data = await response.json();

    await storage.setTokens({
      accessToken: data.access_token,
      refreshToken: data.refresh_token,
    });

    return data;
  }

  /**
   * Create a bookmark
   */
  async createBookmark(bookmarkData) {
    return this.request('/bookmarks', {
      method: 'POST',
      body: JSON.stringify({
        ...bookmarkData,
        source: 'extension',
      }),
    });
  }

  /**
   * Get current user info
   */
  async getMe() {
    return this.request('/auth/me');
  }

  /**
   * Logout - clear local tokens
   */
  async logout() {
    await storage.clearAuth();
  }

  /**
   * Parse JWT token payload
   */
  parseJwt(token) {
    try {
      const base64Url = token.split('.')[1];
      const base64 = base64Url.replace(/-/g, '+').replace(/_/g, '/');
      const jsonPayload = decodeURIComponent(
        atob(base64)
          .split('')
          .map(c => '%' + ('00' + c.charCodeAt(0).toString(16)).slice(-2))
          .join('')
      );
      return JSON.parse(jsonPayload);
    } catch {
      return {};
    }
  }
}

export const api = new ApiClient();
export default api;
