/**
 * Internationalization helper for browser extension
 * Uses Chrome's built-in i18n API
 */

/**
 * Get localized message
 * @param {string} messageName - The message key
 * @param {string|string[]} substitutions - Optional substitutions
 * @returns {string} Localized message
 */
export function getMessage(messageName, substitutions = []) {
  if (typeof chrome !== 'undefined' && chrome.i18n) {
    const subs = Array.isArray(substitutions) ? substitutions : [substitutions];
    return chrome.i18n.getMessage(messageName, subs) || messageName;
  }
  return messageName;
}

/**
 * Get current UI language
 * @returns {string} Language code (e.g., 'en', 'ru')
 */
export function getUILanguage() {
  if (typeof chrome !== 'undefined' && chrome.i18n) {
    return chrome.i18n.getUILanguage().split('-')[0];
  }
  return navigator.language?.split('-')[0] || 'en';
}

/**
 * Check if current language is RTL
 * @returns {boolean}
 */
export function isRTL() {
  const rtlLanguages = ['ar', 'he', 'fa', 'ur'];
  return rtlLanguages.includes(getUILanguage());
}

export default { getMessage, getUILanguage, isRTL };
