/**
 * Storage abstraction for browser extension
 * Uses chrome.storage.sync for cross-device sync
 */

const STORAGE_KEYS = {
  TOKENS: 'sortula_tokens',
  USER: 'sortula_user',
  SETTINGS: 'sortula_settings',
  OFFLINE_QUEUE: 'sortula_offline_queue',
};

class Storage {
  /**
   * Get authentication tokens
   */
  async getTokens() {
    const data = await this.get(STORAGE_KEYS.TOKENS);
    return data || null;
  }

  /**
   * Set authentication tokens
   */
  async setTokens(tokens) {
    await this.set(STORAGE_KEYS.TOKENS, tokens);
  }

  /**
   * Get user info
   */
  async getUser() {
    const data = await this.get(STORAGE_KEYS.USER);
    return data || null;
  }

  /**
   * Set user info
   */
  async setUser(user) {
    await this.set(STORAGE_KEYS.USER, user);
  }

  /**
   * Get extension settings
   */
  async getSettings() {
    const data = await this.get(STORAGE_KEYS.SETTINGS);
    return {
      language: 'auto',
      notifications: true,
      ...data,
    };
  }

  /**
   * Update extension settings
   */
  async setSettings(settings) {
    const current = await this.getSettings();
    await this.set(STORAGE_KEYS.SETTINGS, { ...current, ...settings });
  }

  /**
   * Get offline queue
   */
  async getOfflineQueue() {
    const data = await this.get(STORAGE_KEYS.OFFLINE_QUEUE);
    return data || [];
  }

  /**
   * Add item to offline queue
   */
  async addToOfflineQueue(item) {
    const queue = await this.getOfflineQueue();
    queue.push({
      ...item,
      id: Date.now().toString(),
      createdAt: new Date().toISOString(),
    });
    await this.set(STORAGE_KEYS.OFFLINE_QUEUE, queue);
    return queue;
  }

  /**
   * Remove item from offline queue
   */
  async removeFromOfflineQueue(itemId) {
    const queue = await this.getOfflineQueue();
    const filtered = queue.filter(item => item.id !== itemId);
    await this.set(STORAGE_KEYS.OFFLINE_QUEUE, filtered);
    return filtered;
  }

  /**
   * Clear offline queue
   */
  async clearOfflineQueue() {
    await this.set(STORAGE_KEYS.OFFLINE_QUEUE, []);
  }

  /**
   * Clear all auth data (logout)
   */
  async clearAuth() {
    await this.remove(STORAGE_KEYS.TOKENS);
    await this.remove(STORAGE_KEYS.USER);
  }

  /**
   * Check if user is authenticated
   */
  async isAuthenticated() {
    const tokens = await this.getTokens();
    return !!tokens?.accessToken;
  }

  // Low-level storage methods using chrome.storage.sync
  async get(key) {
    return new Promise((resolve) => {
      if (typeof chrome !== 'undefined' && chrome.storage) {
        chrome.storage.sync.get([key], (result) => {
          resolve(result[key]);
        });
      } else if (typeof browser !== 'undefined' && browser.storage) {
        browser.storage.sync.get([key]).then((result) => {
          resolve(result[key]);
        });
      } else {
        // Fallback to localStorage for development
        const value = localStorage.getItem(key);
        resolve(value ? JSON.parse(value) : undefined);
      }
    });
  }

  async set(key, value) {
    return new Promise((resolve) => {
      if (typeof chrome !== 'undefined' && chrome.storage) {
        chrome.storage.sync.set({ [key]: value }, resolve);
      } else if (typeof browser !== 'undefined' && browser.storage) {
        browser.storage.sync.set({ [key]: value }).then(resolve);
      } else {
        // Fallback to localStorage for development
        localStorage.setItem(key, JSON.stringify(value));
        resolve();
      }
    });
  }

  async remove(key) {
    return new Promise((resolve) => {
      if (typeof chrome !== 'undefined' && chrome.storage) {
        chrome.storage.sync.remove([key], resolve);
      } else if (typeof browser !== 'undefined' && browser.storage) {
        browser.storage.sync.remove([key]).then(resolve);
      } else {
        // Fallback to localStorage for development
        localStorage.removeItem(key);
        resolve();
      }
    });
  }
}

export const storage = new Storage();
export default storage;
